﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/ce/CostExplorerRequest.h>
#include <aws/ce/CostExplorer_EXPORTS.h>
#include <aws/ce/model/DateInterval.h>
#include <aws/ce/model/Expression.h>
#include <aws/ce/model/Granularity.h>
#include <aws/ce/model/GroupDefinition.h>
#include <aws/ce/model/SortDefinition.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CostExplorer {
namespace Model {

/**
 */
class GetSavingsPlansCoverageRequest : public CostExplorerRequest {
 public:
  AWS_COSTEXPLORER_API GetSavingsPlansCoverageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetSavingsPlansCoverage"; }

  AWS_COSTEXPLORER_API Aws::String SerializePayload() const override;

  AWS_COSTEXPLORER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The time period that you want the usage and costs for. The <code>Start</code>
   * date must be within 13 months. The <code>End</code> date must be after the
   * <code>Start</code> date, and before the current date. Future dates can't be used
   * as an <code>End</code> date.</p>
   */
  inline const DateInterval& GetTimePeriod() const { return m_timePeriod; }
  inline bool TimePeriodHasBeenSet() const { return m_timePeriodHasBeenSet; }
  template <typename TimePeriodT = DateInterval>
  void SetTimePeriod(TimePeriodT&& value) {
    m_timePeriodHasBeenSet = true;
    m_timePeriod = std::forward<TimePeriodT>(value);
  }
  template <typename TimePeriodT = DateInterval>
  GetSavingsPlansCoverageRequest& WithTimePeriod(TimePeriodT&& value) {
    SetTimePeriod(std::forward<TimePeriodT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>You can group the data using the attributes <code>INSTANCE_FAMILY</code>,
   * <code>REGION</code>, or <code>SERVICE</code>.</p>
   */
  inline const Aws::Vector<GroupDefinition>& GetGroupBy() const { return m_groupBy; }
  inline bool GroupByHasBeenSet() const { return m_groupByHasBeenSet; }
  template <typename GroupByT = Aws::Vector<GroupDefinition>>
  void SetGroupBy(GroupByT&& value) {
    m_groupByHasBeenSet = true;
    m_groupBy = std::forward<GroupByT>(value);
  }
  template <typename GroupByT = Aws::Vector<GroupDefinition>>
  GetSavingsPlansCoverageRequest& WithGroupBy(GroupByT&& value) {
    SetGroupBy(std::forward<GroupByT>(value));
    return *this;
  }
  template <typename GroupByT = GroupDefinition>
  GetSavingsPlansCoverageRequest& AddGroupBy(GroupByT&& value) {
    m_groupByHasBeenSet = true;
    m_groupBy.emplace_back(std::forward<GroupByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The granularity of the Amazon Web Services cost data for your Savings Plans.
   * <code>Granularity</code> can't be set if <code>GroupBy</code> is set.</p> <p>The
   * <code>GetSavingsPlansCoverage</code> operation supports only <code>DAILY</code>
   * and <code>MONTHLY</code> granularities.</p>
   */
  inline Granularity GetGranularity() const { return m_granularity; }
  inline bool GranularityHasBeenSet() const { return m_granularityHasBeenSet; }
  inline void SetGranularity(Granularity value) {
    m_granularityHasBeenSet = true;
    m_granularity = value;
  }
  inline GetSavingsPlansCoverageRequest& WithGranularity(Granularity value) {
    SetGranularity(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Filters Savings Plans coverage data by dimensions. You can filter data for
   * Savings Plans usage with the following dimensions:</p> <ul> <li> <p>
   * <code>LINKED_ACCOUNT</code> </p> </li> <li> <p> <code>REGION</code> </p> </li>
   * <li> <p> <code>SERVICE</code> </p> </li> <li> <p> <code>INSTANCE_FAMILY</code>
   * </p> </li> </ul> <p> <code>GetSavingsPlansCoverage</code> uses the same <a
   * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html">Expression</a>
   * object as the other operations, but only <code>AND</code> is supported among
   * each dimension. If there are multiple values for a dimension, they are OR'd
   * together.</p> <p>Cost category is also supported.</p>
   */
  inline const Expression& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = Expression>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = Expression>
  GetSavingsPlansCoverageRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The measurement that you want your Savings Plans coverage reported in. The
   * only valid value is <code>SpendCoveredBySavingsPlans</code>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetMetrics() const { return m_metrics; }
  inline bool MetricsHasBeenSet() const { return m_metricsHasBeenSet; }
  template <typename MetricsT = Aws::Vector<Aws::String>>
  void SetMetrics(MetricsT&& value) {
    m_metricsHasBeenSet = true;
    m_metrics = std::forward<MetricsT>(value);
  }
  template <typename MetricsT = Aws::Vector<Aws::String>>
  GetSavingsPlansCoverageRequest& WithMetrics(MetricsT&& value) {
    SetMetrics(std::forward<MetricsT>(value));
    return *this;
  }
  template <typename MetricsT = Aws::String>
  GetSavingsPlansCoverageRequest& AddMetrics(MetricsT&& value) {
    m_metricsHasBeenSet = true;
    m_metrics.emplace_back(std::forward<MetricsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to retrieve the next set of results. Amazon Web Services provides
   * the token when the response from a previous call has more results than the
   * maximum page size.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  GetSavingsPlansCoverageRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of items to be returned in a response. The default is
   * <code>20</code>, with a minimum value of <code>1</code>.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline GetSavingsPlansCoverageRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value that you want to sort the data by.</p> <p>The following values are
   * supported for <code>Key</code>:</p> <ul> <li> <p>
   * <code>SpendCoveredBySavingsPlan</code> </p> </li> <li> <p>
   * <code>OnDemandCost</code> </p> </li> <li> <p> <code>CoveragePercentage</code>
   * </p> </li> <li> <p> <code>TotalCost</code> </p> </li> <li> <p>
   * <code>InstanceFamily</code> </p> </li> <li> <p> <code>Region</code> </p> </li>
   * <li> <p> <code>Service</code> </p> </li> </ul> <p>The supported values for
   * <code>SortOrder</code> are <code>ASCENDING</code> and
   * <code>DESCENDING</code>.</p>
   */
  inline const SortDefinition& GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  template <typename SortByT = SortDefinition>
  void SetSortBy(SortByT&& value) {
    m_sortByHasBeenSet = true;
    m_sortBy = std::forward<SortByT>(value);
  }
  template <typename SortByT = SortDefinition>
  GetSavingsPlansCoverageRequest& WithSortBy(SortByT&& value) {
    SetSortBy(std::forward<SortByT>(value));
    return *this;
  }
  ///@}
 private:
  DateInterval m_timePeriod;
  bool m_timePeriodHasBeenSet = false;

  Aws::Vector<GroupDefinition> m_groupBy;
  bool m_groupByHasBeenSet = false;

  Granularity m_granularity{Granularity::NOT_SET};
  bool m_granularityHasBeenSet = false;

  Expression m_filter;
  bool m_filterHasBeenSet = false;

  Aws::Vector<Aws::String> m_metrics;
  bool m_metricsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  SortDefinition m_sortBy;
  bool m_sortByHasBeenSet = false;
};

}  // namespace Model
}  // namespace CostExplorer
}  // namespace Aws
